﻿' 版权所有 (C) Microsoft Corporation。保留所有权利。
Imports System.Text.RegularExpressions
Imports System.IO
Imports System.Net

Public Class MainForm

    Private images As System.Collections.Generic.Dictionary(Of String, ImageAttributes)
    Protected viewerForm As ImageViewer
    Private statusForm As Status
    Public domain As String
    Public urlEntered As String
    Private requestScrape As HttpWebRequest
    Private responseScrape As HttpWebResponse

    ''' <summary>
    ''' 处理“RegEx Tester”选项卡上的“Find”（查找）按钮的
    ''' Click 事件。此例程在源文本中查找匹配项，显示组，并
    ''' 捕获结果 TextBox 中的值。
    ''' </summary>
    Private Sub btnFind_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnFind.Click
        Dim expression As Regex
        Dim options As RegexOptions = GetRegexOptions()

        Try
            expression = New Regex(txtRegEx.Text, options)
        Catch exp As Exception
            MsgBox("An error was encountered when attempting to parse the " & _
                "source text. This can be caused by an invalid regex pattern." & _
                "  Check your expression and try again." & vbCrLf & exp.Message, _
                MsgBoxStyle.Critical, Me.Text)
            Exit Sub
        End Try

        ' 获取将模式应用到源文本后产生的
        ' 匹配项的集合。
        Dim matches As MatchCollection = expression.Matches(txtSource.Text)

        ' 显示匹配项的数量并清除所有现有结果。
        lblResultCount.Text = matches.Count.ToString & " matches"
        txtResults.Clear()

        ' 循环访问匹配项集合，并根据 UI 设置
        ' 显示组和/或捕获的值。
        For Each m As Match In matches
            AppendResults("'" & m.Value & "' at index " & m.Index)

            If m.Groups.Count > 1 And chkShowGroups.Checked Then
                Dim g As Group
                ' 跳过第 0 组，它是整个 Match 对象。
                For i As Integer = 1 To m.Groups.Count - 1
                    ' 获取对相应组的引用。
                    g = m.Groups(i)

                    AppendResults(String.Format("   group({0}) = '{1}'", i, g.Value))

                    If chkShowCaptures.Checked = True Then
                        ' 获取此组的捕获集合。
                        Dim captures As CaptureCollection = g.Captures
                        ' 显示有关每个捕获的信息。
                        For Each c As Capture In captures
                            AppendResults(String.Format("      " & _
                                "capture '{0}' at index {1}", c.Value, c.Index))
                        Next
                    End If
                Next
            End If
        Next
    End Sub

    ' 此例程处理“Find the Number”（查找数字）按钮的 Click 事件。
    ' 它查找字符串中的第一个数字，并显示该数字
    ' 和它在字符串中的起始位置。
    Private Sub btnFindNumber_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnFindNumber.Click
        ' 创建 RegEx 的一个实例，传入模式，模式将查找一个或
        ' 多个数字。
        Dim expression As New Regex("\d+")
        ' 调用 Match，传入源文本。
        Dim m As Match = expression.Match(txtFindNumber.Text)

        ' 如果找到匹配，则显示结果。如果未找到，则显示一条“错误”信息。
        If m.Success Then
            lblResults.Text = "RegEx found " & m.Value & " at position " & m.Index.ToString
        Else
            lblResults.Text = "You didn't enter a string containing a number."
        End If
    End Sub

    ' 处理“Replace”（替换）按钮的 Click 事件。当找到匹配时，
    ' 用替换文本替换源文本。
    Private Sub btnReplace_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnReplace.Click
        Try
            txtResults.Text = Regex.Replace(txtSource.Text, txtRegEx.Text, _
                txtReplace.Text, GetRegexOptions())
        Catch exp As Exception
            MsgBox(exp.ToString, MsgBoxStyle.Critical, Me.Text)
        End Try
    End Sub

    ' 处理“Scrape”（擦除）按钮的 Click 事件。
    Private Sub btnScrape_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnScrape.Click
        Dim httpSource As String

        If IsUrlValid() Then
            ' 获取一个包含网页源代码的字符串。
            ' 有关更多注释，请参见此处使用的两个自定义函数。
            Try
                httpSource = ConvertStreamToString(GetHttpStream(txtURL.Text))
            Catch exp As Exception
                ' 将文本设置为 exp.Message 以显示在调用的函数中
                ' 设置的自定义错误信息。 
                MsgBox(exp.Message, MsgBoxStyle.Critical, Me.Text)
                Exit Sub
            End Try

            Dim regExPattern As String

            ' 为用户输入的完全 Url 分配一个变量。
            urlEntered = Trim(txtURL.Text)
            ' 获取用于不同位置的域名。最好在
            ' Click 事件中设置值，以使在任一 ListView 控件中显示的项
            ' 总和正确的域名关联。如果在其他地方
            ' 分配该值，则用户能更改域名并在 ListView 中双击
            ' 某项，这可能会导致此项的
            ' Url 无效。
            Dim astrDomain() As String = Regex.Split(urlEntered, "/")
            ' 第一个元素 (0) 是“http:”，而第三个元素是
            ' 实际域名。
            domain = astrDomain(0) & "//" & astrDomain(2)

            If optLinks.Checked Then
                ShowStatusIndicators("Connecting to Web page to screen scrape " & _
                    "the links. Please stand by...")

                lvwLinks.Items.Clear()

                ' 此处用于分析 HTML 定位点的正则表达式，如，
                ' <a href="http://www.microsoft.com/net">Microsoft</a> 的解释
                ' 如下：
                '   <a          HTML 定位点的开始
                '   \s+         一个或多个空白字符
                '   href        后接 HTML 定位点中的确切文本
                '   \s*         零个或多个空白字符
                '   =           后接 HTML 定位点中的确切文本
                '   \s*         零个或多个空白字符
                '   ""?         零或无引号（转义的）
                '   (           定义子字符串（定位点 URL）的组的起始点。
                '   [^"" >]+    任意字符的一个或多个匹配项，
                '               括号中的字符除外。
                '   )           定义子字符串的第一组的结束
                '   ""?         零或无引号（转义的）
                '   >           后接 HTML 定位点中的确切文本
                '   (.+)        与任意字符（定位点文本）匹配的组。
                '   </a>       结束 HTML 定位点的确切文本               
                '
                ' 警告：如果要在 RegEx Tester 中使用这些模式，
                ' 请确保没有转义双引号。
                regExPattern = "<a\s+href\s*=\s*""?([^"" >]+)""?>(.+)</a>"
            Else
                ShowStatusIndicators("Connecting to Web page to screen scrape the " & _
                    "images. Please stand by...")

                images = New System.Collections.Generic.Dictionary(Of String, ImageAttributes)
                lvwImages.Items.Clear()

                ' 擦除图像的正则表达式在概念上类似于
                ' 擦除 HTML 定位点的模式。然而，在这种情况下，
                ' 模式在概念上更为复杂，因为我们要捕获多达
                ' 四种的不同属性，且这些属性能以不同顺序出现。
                '
                ' 为了使此处使用的正则表达式的长度和复杂性保持
                ' 合理，做出下列关于在任何给定网页上
                ' 处理的 HTML <img> 标记的假设：
                '
                '   1. src 属性总是出现。它是唯一必需的
                '      属性。
                '   2. src 属性在 width 和 height 之前。否则，不会抓取
                '      width 和 height。
                '   3. alt 属性在 width 和 height 之后。否则，不会
                '      抓取 alt。
                '   4. Width 和 height 在 src 之后，可采用任意相对顺序。
                '      模式包括了这两种选项。
                '
                ' 这确保了抓取页上所有图像。但这意味着
                ' 一些其他属性数据可能不会出现。
                '
                ' 此模式中使用的一些关键短语是：
                ' 
                '   [^>]+       与除 > 之外的任意字符匹配。这是
                '               移动到感兴趣的下一个属性的方式。
                '   (?:         启动一个非捕获组。和
                '               关闭 )? 一起使用可创建可选组（零个或一个
                '               匹配）。这是使除 src 之外的所有属性可选
                '               的方式。
                '   |           与 width 和 height 一起用作“或”表达式。注意
                '               在第一对中 width 在前，而在
                '               第二对中，顺序反转。
                regExPattern = "<img[^>]+(src)\s*=\s*""?([^ "">]+)""?(?:[^>]+(width|height)\s*=\s*""?([^ "">]+)""?\s+(height|width)\s*=\s*""?([^ "">]+)""?)?(?:[^>]+(alt)\s*=\s*""?([^"">]+)""?)?"
            End If

            Dim re As New Regex(regExPattern, RegexOptions.IgnoreCase)
            Dim m As Match = re.Match(httpSource)

            ' 处理 HTML 源代码。因为源是长字符串，
            ' 所以使用更高效的 Match() 方法而非 Matches 
            ' 方法，前者一次只返回一个匹配。Success 属性确定
            ' 是否存在另一匹配。NextMatch() 引发迭代。
            While m.Success
                If optImages.Checked Then

                    Dim width As String
                    Dim height As String
                    ' 因为模式就 width 属性和 height 属性给出了可选顺序，
                    ' 请确定哪个属性列在前面，
                    ' 然后分配正确的 Group 项值。
                    If m.Groups(3).Value.ToLower = "width" Then
                        width = m.Groups(4).Value
                        height = m.Groups(6).Value
                    Else ' Height 属性在前
                        width = m.Groups(6).Value
                        height = m.Groups(4).Value
                    End If

                    ' 调用 AddImage 以向哈希表添加一个自定义 ImageAttributes
                    ' 对象的实例。有关为何使用哈希表的更多注释，
                    ' 请参见 AddImage。
                    AddImage(New ImageAttributes(m.Groups(2).Value, _
                        m.Groups(8).Value, height, width))
                Else
                    ' 创建一个 ListViewItem 对象，并设置第一列的
                    ' 文本（“src”）。
                    Dim lvi As New ListViewItem()
                    lvi.Text = m.Groups(1).Value
                    lvwLinks.Items.Add(lvi)
                End If

                ' 继续循环到下一个匹配项。
                m = m.NextMatch()
            End While

            HideStatusIndicators()

            ' 显示控件相应的结果。
            If optImages.Checked Then
                If images IsNot Nothing Then
                    FillImagesListView()
                Else
                    MsgBox("No images were found whose attributes matched the " & _
                        "regular expression.", MsgBoxStyle.Information, Me.Text)
                End If
            Else
                If lvwLinks.Items.Count = 0 Then
                    MsgBox("No links were found whose Url matched the regular " & _
                    "expression.", MsgBoxStyle.Information, Me.Text)
                End If
            End If
        End If
    End Sub

    ' 处理“Split”（拆分）按钮的 Click 事件。此例程将源文本拆分为字符串数组
    ' 而非 MatchCollection（如在“Find”（查找）按钮中）。
    ' RegEx.Split 与 String.Split 类似，不同的是它使用正则表达式而非
    ' 单个字符来定义分隔符。
    Private Sub btnSplit_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnSplit.Click
        ' 清除现有结果并拆分源文本。
        txtResults.Clear()

        ' 此处我们调用共享的 Split 方法，不使用 RegEx 实例。
        ' 可尝试 \s*,\s* 这个较好的拆分模式，它将
        ' 创建由逗号分隔的任意字符的数组。在这种情况下，
        ' RegEx.Split(txtSource.Text, "\s*,\s*") 将与下面的语句产生相同的结果：
        '
        '       txtRegEx.Text.Split(New Char() {CChar(",")})
        '       Microsoft.VisualBasic.Split(txtRegEx.Text, ",")
        '
        ' 您还可以进一步修改正则表达式以丢弃结果字符串数组中的任意空
        ' 元素：\s*[,]+\s*
        Dim results() As String
        Try
            results = Regex.Split(txtSource.Text, txtRegEx.Text, _
                GetRegexOptions())
        Catch exp As Exception
            MsgBox("An error was encountered when attempting to parse the " & _
                "source text. This can be caused by an invalid regex pattern." & _
                "  Check your expression and try again." & vbCrLf & exp.Message, _
                MsgBoxStyle.Critical, Me.Text)
            Exit Sub
        End Try

        ' 要进行拆分，必须创建至少有两个元素的字符串数组。
        If results.Length > 1 Then
            AppendResults("The source text was split into " & _
                results.Length & " items.")

            For Each s As String In results
                AppendResults(s)
            Next
        Else
            AppendResults("The source text could not be split. " & _
                "Check your regular expression pattern and try again.")
        End If
    End Sub

    ' 处理“Simple Samples”（简单示例）选项卡上的“Validate!”（验证！）按钮的 Click 事件。
    Private Sub btnValidate_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnValidate.Click
        Dim isValid As Boolean = True

        ' 当执行验证时，通常始终使用
        ' 一组模式元素：
        '
        '  ^ 和 $     源字符串的内容只能是这两个元素之间的字符，即日期。
        ' 最好在这些字符中包装验证模式，
        ' 否则将仅在模式出现处进行匹配。
        ' 例如，如果不包装验证模式， 
        ' 这将在应用程序中作为有效日期传递：“kjdi12-25-2000xpL” 
        '
        ' ^ 和 $     
        ' ^ 字符与输入字符串起始位置相匹配。
        ' $ 字符与输入字符串结束位置相匹配。
        ' 最好在这些字符串中包装验证模式，
        ' 这样仅当整个字符串与模式相匹配时字符串才通过验证。
        ' 否则，字符串的任意子字符串匹配都将报告为匹配。
        ' 例如，假设有一个验证日期的正则表达式，
        ' “12-25-2000”将通过验证。如果正则表达式不是以
        ' ^ 开始并以 $ 结束，则字符串“kjdi12-25-2000xpL”将同样通过验证。

        '   \s*         位于字符串的起始和结束位置，它指示
        '               可接受空白。
        '
        ' 下面的模式检查输入字符串是否为有效的邮政编码，有效邮政编码的
        ' 格式为 ddddd 或 ddddd-dddd，其中 d 是 0-9 的任何数字。
        ' 使用的主要模式元素为：
        '
        '   \d          任意数字 (0-9)。
        '   |           管道符表示邮政编码既可以是 5 个数字
        '               也可以是 5 个数字后面跟有一短划线和 4 个数字。
        '
        If Not Regex.IsMatch(txtZip.Text, "^\s*(\d{5}|(\d{5}-\d{4}))\s*$") Then
            txtZip.ForeColor = Color.Red
            isValid = False
        Else
            txtZip.ForeColor = Color.Black
        End If

        ' 下面的模式检查输入字符串是否为
        ' mm-dd-yy 格式或 mm-dd-yyyy 格式的日期。 使用的主要模式元素为：
        '
        '   \d{1,2}     月和日数可有 1 或 2 个数字。
        '               使用 (\d{4}|\d{2}) 意味着年可有 2 或 4 个数字。
        '   (/|-)       斜杠或短划线都是有效的日期分隔符。
        '   \1          用于日和年的分隔符必须与用于月和日
        '               的分隔符相同。1 引用第一个用括号定义的组，
        '               如 (/|-)。
        '   
        ' 可通过确保数字不从零开始并且
        ' 处于有效数值范围内来改善此模式。
        '
        If Not Regex.IsMatch(txtDate.Text, _
            "^\s*\d{1,2}(/|-)\d{1,2}\1(\d{4}|\d{2})\s*$") Then

            txtDate.ForeColor = Color.Red
            isValid = False
        Else
            txtDate.ForeColor = Color.Black
        End If

        ' 下面的模式检查输入字符串是否为有效的、形式
        ' 为“name@domain.com”的电子邮件地址。实际上，
        ' 不是必须为“.com”地址。最后一个点号后可以是任意字母
        ' 组合。同样，电子邮件名可包含短划线或由一个或多个
        ' 点号分隔。域名同样可包含由点号分隔的多个单词。

        ' [\w-]+    
        ' 一个或多个任意字符（a-z、A-Z、0-9 以及下划线) 或短划线。在 @ 字符两边，这确保地址形式为 name@domainname。
        ' \.              
        ' 一个转义点号。（不带反斜杠，一个点号与除换行符外的任意单个字符匹配。）以此确保域名中至少有一个点号。
        ' *?            
        ' 对前面的表达式，非贪婪 (non-greedy，即最小)地查找零次或多次匹配。
        ' ([\w-]+\.)*?    
        ' 以上三个表达式的组合：
        ' 对于包含一个或多个任意字符（a-z、A-Z、0-9 以及下划线）或短划线并且后面只跟一个点号的表达式，非贪婪地查找零次或多次匹配。

        If Not Regex.IsMatch(txtEmail.Text, _
            "^([\w-]+\.)*?[\w-]+@[\w-]+\.([\w-]+\.)*?[\w]+$") Then

            txtEmail.ForeColor = Color.Red
            isValid = False
        Else
            txtEmail.ForeColor = Color.Black
        End If

        If isValid Then
            lblValid.Visible = True
        Else
            lblValid.Visible = False
        End If
    End Sub

    ' 此例程处理“Images”（图像）列表视图的 DoubleClick 双击事件。
    ' 双击一个图像打开 frmImageViewer，按流从 Web 服务器请求图像，
    ' 并在 PictureBox 中显示该图像。
    Private Sub lvImages_DoubleClick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles lvwImages.DoubleClick

        ' 在验证检查中包括所有代码，确保用户没有在双击项之前
        ' 将 Url 更改为无效值。
        If IsUrlValid() Then
            Dim imageSource As String = lvwImages.SelectedItems(0).Text

            ShowStatusIndicators("Connecting to Web page to retrieve the " & _
                "selected image. Please stand by...")

            If IsNothing(viewerForm) Then
                viewerForm = New ImageViewer()
            End If

            ' 如果用户单击的图像路径不是绝对路径，则
            ' 更正它并显示 ImageViewer 窗体。
            If Microsoft.VisualBasic.Left(imageSource, 7) <> "http://" Then
                imageSource = MakeRelativeUrlAbsolute(imageSource)
            End If

            viewerForm.Show(imageSource)

            HideStatusIndicators()
        End If

    End Sub

    ' 此例程处理“Links”（链接）ListView 的 DoubleClick 事件。 
    ' 双击链接启动 Internet Explorer 的新实例，并
    ' 导航到所请求的页。
    Private Sub lvLinks_DoubleClick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles lvwLinks.DoubleClick

        ' 在验证检查中包括所有代码，确保用户没有在双击项之前
        ' 将 Url 更改为无效值。
        If IsUrlValid() Then

            Dim clickedUrl As String = lvwLinks.SelectedItems(0).Text

            ShowStatusIndicators("Starting Internet Explorer and connecting to " & _
                "the selected Web page. Please stand by...")

            ' 如果用户单击的图像路径不是绝对路径，则 
            ' 更正它并启动 Internet Explorer，导航到该页。
            If Microsoft.VisualBasic.Left(clickedUrl, 7) <> "http://" Then
                clickedUrl = MakeRelativeUrlAbsolute(clickedUrl)
            End If

            ' 它是相对于根目录的路径或相对路径。所以，请将
            ' 相对 Url 更改为绝对 Url。
            Process.Start("IExplore.exe", clickedUrl)

            HideStatusIndicators()

        End If

    End Sub

    ' 此例程处理“Screen Scrape”（屏幕擦除）选项卡上 RadioButton 控件的 CheckedChanged 事件。
    ' 根据选中的选项将 ListView 控件
    ' 交换调出。
    Private Sub RadioButtons_CheckedChanged(ByVal sender As Object, ByVal e As System.EventArgs) Handles optImages.CheckedChanged, optLinks.CheckedChanged
        If optLinks.Checked Then
            lvwLinks.Visible = True
            lvwImages.Visible = False
        Else
            lvwImages.Visible = True
            lvwLinks.Visible = False
        End If
    End Sub

    ' 此例程向哈希表添加一个 ImageAttributes 对象（如果还没有添加）。
    ' “Src”属性用作查找的键。
    ' 如果 Src 键已经存在，则不添加对象。这可阻止相同
    ' 图像在列表中出现多次。
    Protected Sub AddImage(ByVal imgAttr As ImageAttributes)
        If Not images.ContainsKey(imgAttr.Src) Then
            images.Add(imgAttr.Src, imgAttr)
        End If
    End Sub

    ' 这是一个用于追加文本结果的帮助器例程。
    Sub AppendResults(ByVal msg As String)
        txtResults.AppendText(msg & ControlChars.CrLf)
    End Sub

    ' 此函数读取由 ttpWebResponse 对象返回的流，并
    ' 将其转换为字符串供 RegEx 处理。
    Function ConvertStreamToString(ByVal stmSource As Stream) As String
        Dim sr As StreamReader = Nothing

        If Not IsNothing(stmSource) Then
            Try
                sr = New StreamReader(stmSource)
                ' 读取并返回流的全部内容。
                Return sr.ReadToEnd
            Catch exp As Exception
                ' 不显示 MsgBox。只是转发
                '来自 GetHttpStream() 的错误信息。
                Throw New Exception()
            Finally
                ' 清理 Stream 和 StreamReader。
                responseScrape.Close()
                sr.Close()
            End Try
        End If
        Return Nothing
    End Function

    ' 此例程循环访问哈希表，并为哈希表中的每个自定义 ImageAttributes 对象
    ' 添加具有 ListViewItem.SubItems 的
    ' ListViewItem。
    Protected Sub FillImagesListView()
        For Each source As String In images.Keys
            ' 创建一个 ListViewItem 对象并设置第一列的文本。
            Dim lvi As New ListViewItem()
            lvi.Text = source

            ' 设置剩余列中的文本，
            ' 并向 ListView 添加 ListViewItem 对象。
            Dim imgAttr As ImageAttributes = CType(images(source), ImageAttributes)
            With lvi.SubItems
                .Add(imgAttr.Alt)
                .Add(imgAttr.Height)
                .Add(imgAttr.Width)
            End With

            ' 向 ListView 的项集合添加 ListViewItem。
            lvwImages.Items.Add(lvi)
        Next
    End Sub

    ' 此函数使用从 System.Net.WebRequest 派生的 .NET 类以 
    ' 检索一个 HTTP 响应流，该流成为 RegEx 分析源或者
    ' 由 frmImageViewer.Show() 调用时要显示的图像。
    Function GetHttpStream(ByVal url As String) As Stream
        ' 使用 WebRequestFactory 创建请求。
        requestScrape = CType(WebRequest.Create(url), HttpWebRequest)
        With requestScrape
            .UserAgent = "Mozilla/4.0 (compatible; MSIE 6.0b; Windows NT 5.1)"
            .Method = "GET"
            .Timeout = 10000
        End With

        Try
            ' 返回响应流。
            responseScrape = CType(requestScrape.GetResponse(), HttpWebResponse)
            Return responseScrape.GetResponseStream()
        Catch exp As Exception
            ' 因为错误最有可能是由键入错误的 Url 或者
            ' 没有 Internet 连接造成的，所以创建一条转发回调用函数的
            ' 自定义错误信息。
            Throw New Exception("There was an error retrieving the Web page " & _
                "you requested. Please check the Url and your connection to " & _
                "the Internet, and try again.")
        End Try
    End Function

    ' 这是一个帮助器例程，用于检索各种正则表达式模式
    ' 匹配选项。
    Function GetRegexOptions() As RegexOptions
        ' 使用任意 RegExOptions 都与将整个正则
        ' 表达式放在括号中（即，使整个模式自成一组）并在
        ' 左括号内的表达式前放置选项
        ' 字符相同。 如果要对选项进行更好的控制 -- 即
        ' 在特定组内 -- 在模式内创建一组并使用相同的语法。
        ' 在这种情况下，您不希望使用任何 RegexOptions，
        ' 因为它们将应用到整个模式。要关闭选项，
        ' 仅需对其求反（如，要关闭区分大小写，您可在组
        ' 的左括号内输入 ?-i:）。
        '
        ' 默认情况下所有选项都是关闭的。
        '
        ' IgnoreCase 枚举打开/关闭整个模式的区分大小写。
        ' 使用此枚举与只使用正则表达式语法
        ' 键入 (?i:OriginalPattern) 是相同的。
        If chkIgnoreCase.Checked Then GetRegexOptions = RegexOptions.IgnoreCase

        ' Singleline 枚举更改 .（点号）字符的行为，
        ' 所以它现在与任意字符都匹配（而非它的默认行为，默认行为
        ' 是与除换行符 \r 或 \n 之外的字符匹配）。使用此枚举与只使用
        ' 正则表达式语法键入 (?i:OriginalPattern) 是相同的。
        ' 
        ' 同时注意当由按位“或”运算符分隔时，
        ' 可使用多个 RegExOption。（或者，您可以只使用正则表达式
        ' 语法，通过在 ? 字符后一起放置多个选项来启用多个选项。
        ' 例如，若要打开 Singleline 模式并忽略大小写，
        ' 可键入 (?si:OriginalPattern)。
        If chkSingleLine.Checked Then
            GetRegexOptions = GetRegexOptions Or RegexOptions.Singleline
        End If

        ' Multiline 枚举更改 ^ 和 $ 字符的行为，
        ' 所以它们现在与行的起始和结束相匹配，而非
        ' 整个字符串的起始和结束。使用此枚举与只使用
        ' 正则表达式语法键入 (?m:OriginalPattern) 相同。
        '
        ' 能够同时启用 Singleline 和 Multiline 模式可能会让人感到迷惑，
        ' 但迷惑可能是由于这些选项令人误解的名字。
        ' 如果您忽略其名字并考虑它们的实际
        ' 效果，就不存在冲突。
        If chkMultiline.Checked Then
            GetRegexOptions = GetRegexOptions Or RegexOptions.Multiline
        End If
    End Function

    ' 此例程关闭由 ShowStatusIndicators 启用的状态指示符。
    Private Sub HideStatusIndicators()
        statusForm.Hide()
        Me.Cursor = Cursors.Default
    End Sub

    ' 此函数采用相对 Url 并将其变为绝对 Url。
    ' 它是“Screen Scrape”（屏幕擦除）选项卡上的 DoubleClick 事件处理程序的帮助器。
    ' “href”或“url”属性可包含绝对路径、相对于根目录的路径或
    ' 相对路径。如果被单击项的路径不是绝对路径，此函数
    ' 会通过在其前面加上域名和斜杠（如果需要）使其成为绝对路径。
    Public Function MakeRelativeUrlAbsolute(ByVal relativeUrl As String) As String

        ' 如果它是一个相对于根目录的路径（具有前导“/”），则需要
        ' 在其前面加上域名。如果它是一个相对 Url，则需要在其
        ' 前面加上完整的 Url，同用户输入一样。

        ' 它是相对于根目录的路径吗？
        If Microsoft.VisualBasic.Left(relativeUrl, 1) = "/" Then
            ' 在相对于根目录的路径前面加上域名。
            Return domain & relativeUrl
        ElseIf Microsoft.VisualBasic.Left(relativeUrl, 3) = "../" Then
            ' 移除点号，并在
            ' 相对于根目录的路径前面加上域名。
            Return domain & _
                Microsoft.VisualBasic.Replace(relativeUrl, "../", "/")
        Else ' 它是相对路径。
            ' 检查用户输入的 Url 是否含有尾随的“/”。如果没有，
            ' 则添加一个。
            If Microsoft.VisualBasic.Right(urlEntered, 1) = "/" Then
                Return urlEntered & relativeUrl
            Else
                Return urlEntered & "/" & relativeUrl
            End If
        End If
    End Function

    ' 此例程通过显示具有消息的小状态窗体和设置
    ' WaitCursor 指示正在执行一项任务以向用户提供反馈。
    Private Sub ShowStatusIndicators(ByVal message As String)
        statusForm = New Status()
        statusForm.Show(message)
        Me.Cursor = Cursors.WaitCursor
    End Sub

    ' 此函数检查用户输入的 Url 是否以
    ' http:// 开始。
    Function IsUrlValid() As Boolean
        If Microsoft.VisualBasic.Left(Trim(txtURL.Text), 7) <> "http://" Then
            MsgBox("The Url must begin with http://.")
            Return False
        End If

        Return True
    End Function

    Private Sub exitToolStripMenuItem_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles exitToolStripMenuItem.Click
        Me.Close()
    End Sub
End Class
